# flask_app.py
import configs
import threading
import time
import signal
import sys
import gc

from flask import Flask, request, jsonify
from waitress import serve

from log_service import logger  # Importing the logger

app = Flask(__name__)

# Create a global threading event for clean thread termination
shutdown_event = threading.Event()

def log_heartbeat(port):
    while True:
        logger.info(f"Service is running on port {port}")  # Log a heartbeat message every minute
        time.sleep(60)  # Sleep for 60 seconds
        
        
def signal_handler(sig, frame):
    """Handle termination signals."""
    logger.info("Termination signal received. Stopping service...")
    shutdown_event.set()  # Signal threads to terminate
    sys.exit(0)  # Exit the application

# Register signal handlers for graceful shutdown
signal.signal(signal.SIGINT, signal_handler)  # Handle Ctrl+C
signal.signal(signal.SIGTERM, signal_handler)  # Handle termination signals


@app.route('/code-execution/python', methods=['POST'])
def execute():
    try:
        # Parse the JSON payload from the request
        data = request.get_json()
        logger.info(f"Data received")

        # Extract the code, function call, and parameters from the JSON
        function_code = data.get('code', '')
        function_call = data.get('function_call', '')
        function_params = data.get('function_params', {})

        # Replace dynamic placeholders in the function call (e.g., {data}) with actual parameters
        for key, value in function_params.items():
            val_to_replace = str(value)
            
            if val_to_replace.startswith(('$$.', '$$$.', 'Error: ')):
                logger.info("Invalid value for key: ", key)
                val_to_replace = 'None'
                
            function_call = function_call.replace(f'{{{key}}}', val_to_replace)

        # Full Python code to execute, which includes the function definition and function call
        full_code = f"{function_code}\n\nresult = {function_call}"

        logger.info(f"Executing code.")
        
        # Prepare a local environment to safely execute the Python code
        local_env = {}

        # Execute the function code and get the result
        exec(full_code, {}, local_env)
        
        result = local_env.pop('result', None)  # Remove result from memory immediately

        logger.info("Code execution completed.")

        del local_env  # Explicitly delete variables
        gc.collect()

        # Log the result
        logger.info(f"Execution result: {result}")

        # Return the result in the JSON response
        return jsonify(result), 200

    except Exception as e:
        # Log the error
        logger.error(f"Error during execution: {str(e)}")
        # Return a 400 Bad Request status with error message
        return jsonify({'error': str(e)}), 400


if __name__ == '__main__':
    configs.load_configs()
    
    logger.info("Starting DvSum Flask app...")
    
    port = configs.flask_port()
    threads = configs.flask_threads()
    
    # Start the heartbeat logging thread
    heartbeat_thread = threading.Thread(target=log_heartbeat, args=(port,), daemon=True)
    heartbeat_thread.start()
    
    try:
        logger.info("Active threads: " + threads)
        # Use Waitress for production-ready serving
        serve(app, host='127.0.0.1', port=port, threads=int(threads))
    finally:
        logger.info("Service stopped.")
        shutdown_event.set()  # Ensure all threads are signaled to terminate