import os
import logging

# Setup logging
logging.basicConfig(level=logging.DEBUG if os.getenv("DEBUG") else logging.INFO)

# Default Configuration
DEFAULT_CONFIGS = {
    'cache.limit': 2048,
    'time.to.live': 600,  # in seconds
    'python.service.port': '8185',
    'python.service.threads': '30',
    'max.file.size': 65
}
CONFIG_FILE_PATH = os.path.join(os.path.dirname(__file__), '..', 'configuration.properties')
loaded_configs = {}

def load_configs():
    """Loads the configuration from the config file."""
    logging.info("Loading configs from %s", CONFIG_FILE_PATH)
    
    if not os.path.exists(CONFIG_FILE_PATH):
        logging.warning("Config file not found, using default configurations.")
        return DEFAULT_CONFIGS
    
    try:
        with open(CONFIG_FILE_PATH, 'r') as config_file:
            for line in config_file:
                line = line.strip()
                
                # Skip empty lines and comments
                if not line or line.startswith('#'):
                    continue
                
                if '=' in line:
                    key, value = line.split('=', 1)
                    key, value = key.strip(), value.strip()
                    if value.endswith("\\:"):  # Handle escaped colon in time values
                        value = value.replace("\\:", ":")
                    loaded_configs[key] = value

    except Exception as e:
        logging.error("Error loading config file: %s", e)
        logging.info("Falling back to default configurations.")
        loaded_configs.update(DEFAULT_CONFIGS)
    
    return loaded_configs

def get_config_value(key, default=None):
    """Helper function to fetch the config value."""
    return loaded_configs.get(key, str(default))

def flask_port():
    """Returns the port for the Flask service."""
    return get_config_value('python.service.port', DEFAULT_CONFIGS['python.service.port'])

def flask_threads():
    return get_config_value('python.service.threads', DEFAULT_CONFIGS['python.service.threads'])

def cache_limit():
    """Returns the cache limit."""
    return int(get_config_value('cache.limit', DEFAULT_CONFIGS['cache.limit']))

def ttl():
    """Returns the time-to-live (TTL) value."""
    return int(get_config_value('time.to.live', DEFAULT_CONFIGS['time.to.live']))

def max_file_size():
    """Returns the maximum file size."""
    return int(get_config_value('max.file.size', DEFAULT_CONFIGS['max.file.size']))

# if __name__ == "__main__":
#     load_configs()
#     print("Flask Port:", flask_port())
#     print("Flask Threads:", flask_threads())
#     print("Cache Limit:", cache_limit())
#     print("TTL:", ttl())
#     print("Max File Size:", max_file_size())
