#!/bin/bash

# DvSum Gateway Installation and Configuration Script
# This script automates the process of setting up SSL for the DvSum Gateway
# and configuring it as a service on Linux machines.

# --- Global Variables ---
interactive=true
ssl_option=""
action=""
encryption_algorithm="RSA"
key_size=""  # Will be set based on the algorithm if not specified
gateway_service_name="dvsum-gateway"
gateway_monitor_name="dvsum-monitor"
gateway_python_service_name="dvsum-python"
api_key=""
port_number=""
max_database_connection_allowed=""
python_service_port_number=""

# Set environment variables
JRE_DIR="$(pwd)/jre1.8.0_202_linux"
DVSUM_JAR="$(pwd)/DvSum.jar"
DVSUM_MONITOR_JAR="$(pwd)/DvSum_monitor.jar"
DVSUM_SERVICE_FILE="/etc/systemd/system/dvsum.service"
DVSUM_MONITOR_SERVICE_FILE="/etc/systemd/system/dvsum_monitor.service"
DVSUM_PYTHON_SERVICE_FILE="/etc/systemd/system/dvsum_python.service"
SSL_CONFIG_FILE="ssl_config.properties"
KEYTOOL="$(pwd)/ssl_tools/openlogic-jre-21.0.3+9/bin/keytool"
FLASK_APP="$(pwd)/python_service/flask_app.py"
REQUIREMENTS_FILE="$(pwd)/python_service/requirements.txt"


export LANG=en_US.UTF-8
export JAVA_TOOL_OPTIONS="-Dfile.encoding=UTF-8"

process_successful=true

# --- Utility Functions ---

create_password_file() {
    temp_pass_file=$(mktemp)
    echo -n "$keystore_pass" > "$temp_pass_file"
    chmod 600 "$temp_pass_file"
}

remove_password_file() {
    if [ -f "$temp_pass_file" ]; then
        rm "$temp_pass_file"
    fi
}

validate_properties() {
    local option=$1
    local missing_props=()

    case $option in
        1)
            local required_props=("domain_name" "company_name" "organizational_unit" "city" "state" "country_code" "alias" "keystore_pass")
            ;;
        2)
            local required_props=("keystore_path" "alias" "keystore_pass" "server_cert_path")
            ;;
        3)
            local required_props=("server_cert_path" "private_key_path" "domain_name" "company_name" "organizational_unit" "city" "state" "country_code" "alias" "keystore_pass")
            ;;
        4)
            local required_props=("keystore_path" "keystore_pass")
            ;;
		*)
            echo "Error: Invalid SSL option:"
			echo "  --ssl-option=<1|2|3|4>    		Specify SSL configuration option:"
			echo "                            		1. Generate a new CSR for CA Signed or self-signed certificate."
			echo "                            		2. Add certificates to an existing keystore file."
			echo "                            		3. Import server certificate and private key into a new keystore."
			echo "                            		4. Use an existing keystore containing the server certificate and private key."
            exit 1
            ;;
    esac

    for prop in "${required_props[@]}"; do
        if [ -z "${!prop}" ]; then
            missing_props+=("$prop")
        fi
    done

    if [ ${#missing_props[@]} -ne 0 ]; then
        echo "Error: The following required properties are missing in ssl_config.properties for SSL option $option:"
        printf " - %s\n" "${missing_props[@]}"
        exit 1
    fi
}

get_gateway_config() {
    local config_file="configuration.properties"
    local props=("api.key" "port" "max.database.connection.allowed" "python.service.port")
    local prompts=("DvSum API Key" "port number (1-65535)" "max Database connections allowed" "python service port number (1-65535)")
    local variables=(api_key port_number max_database_connection_allowed python_service_port_number)

    echo
    echo "Gateway Configuration"
    echo "---------------------"

    # Read existing properties into variables
    declare -A existing_properties
    if [ -f "$config_file" ]; then
        while IFS='=' read -r key value; do
            key=$(echo "$key" | sed 's/^[ \t]*//;s/[ \t]*$//')   # Trim whitespace
            value=$(echo "$value" | sed 's/^[ \t]*//;s/[ \t]*$//') # Trim whitespace
            existing_properties["$key"]="$value"
        done < "$config_file"
    else
        echo "Error: Configuration file not found."
        exit 1
    fi

    # Initialize variables for the specific properties we handle
    for i in "${!props[@]}"; do
        local key=${props[$i]}
        local var=${variables[$i]}
        eval "$var=\"${existing_properties[$key]}\""
        [ -z "${!var}" ] && eval "$var=\"(empty)\""  # Default for empty values
    done

    # Function to validate input
    validate_input() {
        local var=$1
        local value=$2
        case "$var" in
            api_key)
                [[ -n "$value" ]]
                ;;
            port_number)
                [[ "$value" =~ ^[0-9]+$ ]] && [ "$value" -ge 1 ] && [ "$value" -le 65535 ]
                ;;
            max_database_connection_allowed)
                [[ "$value" =~ ^[0-9]+$ ]] && [ "$value" -ge 1 ]
                ;;
            python_service_port_number)
                [[ "$value" =~ ^[0-9]+$ ]] && [ "$value" -ge 1 ] && [ "$value" -le 65535 ] && [ "$value" -ne "$port_number" ]
                ;;
        esac
    }

    # Prompt user for each property
    for i in "${!props[@]}"; do
        local prop=${props[$i]}
        local prompt=${prompts[$i]}
        local var=${variables[$i]}

        if [ -n "${!var}" ]; then
            if [ "${!var}" == "(empty)" ]; then
                read -p "Current $prompt is empty. Do you want to set it? (y/n): " change
            else
                read -p "Current $prompt is ${!var}. Do you want to change it? (y/n): " change
            fi
            if [[ "${change,,}" == "y" ]]; then
                unset $var
            fi
        fi

        while [ -z "${!var}" ]; do
            read -p "Enter $prompt: " input
            # Additional validation for `python_service_port_number`
            if [ "$var" == "python_service_port_number" ] && [ "$input" == "$port_number" ]; then
                echo "Error: Python service port cannot be the same as the main port. Please try again."
                continue
            fi
            if validate_input "$var" "$input"; then
                eval "$var='$input'"
            else
                echo "Error: Invalid input. Please try again."
            fi
        done
    done

    # Update only the specific properties in the configuration file
    {
        for key in "${!existing_properties[@]}"; do
            # Check if the property is one of the specific ones we handle
            case "$key" in
                api.key) echo "$key=$api_key" ;;
                port) echo "$key=$port_number" ;;
                max.database.connection.allowed) echo "$key=$max_database_connection_allowed" ;;
                python.service.port) echo "$key=$python_service_port_number" ;;
                *) echo "$key=${existing_properties[$key]}" ;;  # Preserve other properties
            esac
        done
    } > "$config_file"
}



check_network_connectivity() {
    echo
    echo "Checking network connectivity to DvSum application ..."
    echo "-------------------------------------------------------"
    
    # Set the output file path
    output_file="$(pwd)/network_health_check_result.txt"
    
    # Call the Java method to perform checks and write output file
    # Redirect both stdout and stderr to /dev/null to suppress all output
    "${JRE_DIR}/bin/java" -cp DvSum.jar com.dvsum.services.NetworkHealthChecker "$output_file" > /dev/null 2>&1
    
    # Check if the output file was created
    if [ ! -f "$output_file" ]; then
        echo "Error: Output file was not created by the network health check."
        return 1
    fi
    
    echo
    
    # Display the contents of the output file
    cat "$output_file"
    
    # Delete the output file after displaying its contents
    rm "$output_file"
    
    return 0
} 
update_api_key() {
    local config_file="configuration.properties"
    local api_key=""

    # Check if configuration.properties exists
    if [ ! -f "$config_file" ]; then
        echo "Error: $config_file not found."
        exit 1
    fi

    # Try to read api.key from configuration.properties and trim whitespace
    api_key=$(grep "^api.key=" "$config_file" | cut -d'=' -f2 | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')

    # If api.key is not set or empty, ask for it
    if [ -z "$api_key" ]; then
        read -p "Enter DvSum API Key: " api_key
        # Trim whitespace from user input
        api_key=$(echo "$api_key" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')

        # Ensure api_key is not empty after trimming
        if [ -z "$api_key" ]; then
            echo "Error: API Key cannot be empty."
            exit 1
        fi

        # Update configuration.properties with the new api.key
        if grep -q "^api.key=" "$config_file"; then
            # If api.key exists, replace it
            sed -i "s/^api.key=.*/api.key=$api_key/" "$config_file"
        else
            # If api.key doesn't exist, add it to the end of the file
            echo "api.key=$api_key" >> "$config_file"
        fi

        echo "API Key has been updated in $config_file"
    else
        echo "API Key is already set in $config_file"
    fi
}

read_properties_file() {
    local file=$SSL_CONFIG_FILE
    if [ ! -f "$file" ]; then
        echo "Error: Properties file not found: $file"
        exit 1
    fi
    while IFS='=' read -r key value; do
        key=$(echo "$key" | sed 's/^[ \t]*//;s/[ \t]*$//')
        value=$(echo "$value" | sed 's/^[ \t]*//;s/[ \t]*$//')
        if [ -n "$key" ] && [[ ! "$key" =~ ^# ]]; then
            eval "$key=\"$value\""
        fi
    done < "$file"

    # Validate properties based on the SSL option
    validate_properties "$ssl_option"
}

validate_file_path() {
    local path="$1"
    local type="$2"
    local extensions="$3"

    while true; do
        if [ ! -f "$path" ]; then
            echo "Error: File '$path' not found. Check file name and extension" >&2
            read -e -p "Please enter a valid path for the $type file: " path
        else
            # Check if the file has any of the valid extensions
            valid_extension=false
            for ext in $extensions; do
                if [[ "$path" == *.$ext ]]; then
                    valid_extension=true
                    break
                fi
            done

            if [ "$valid_extension" = true ]; then
                echo "$path"
                return
            else
                echo "Error: The $type file must have one of these extensions: $extensions" >&2
                read -p "Please enter a valid path for the $type file (with correct extension): " path
            fi
        fi
    done
}

validate_directory_path() {
    local path="$1"
    local type="$2"

    while true; do
        if [ ! -d "$path" ]; then
            echo "Error: The specified $type directory does not exist."
            read -p "Please enter a valid path for the $type directory: " path
        else
            break
        fi
    done
    echo "$path"
}

# --- User Interface Functions ---

display_welcome_message() {
    echo
    echo "Welcome to the DvSum Gateway Installation and Configuration Wizard!"
    echo "-------------------------------------------------------------------"
    echo "This wizard will guide you through the process of:"
    echo "  1. Setting up SSL for secure communication"
    echo "  2. Installing and configuring the DvSum Gateway as a service on your Linux machine"
    echo
    echo "You'll be asked to provide information for:"
    echo "  • SSL certificate details"
    echo "  • Keystore configuration"
    echo "  • Service setup preferences"
    echo
    echo "Please ensure you have the necessary SSL certificates and server information ready."
    echo "If you're unsure about any step, please consult the DvSum Gateway documentation."
    echo
    echo "Let's get started!"
    echo
}

display_ssl_menu() {
    echo "SSL Configuration Options:"
    echo "-------------------------"
    echo "1. Generate a new certificate:"
    echo "   • Create a Certificate Signing Request (CSR) for a CA-signed certificate"
    echo "   • Or generate a self-signed certificate for testing"
    echo
    echo "2. Use an existing keystore:"
    echo "   • Add your SSL certificate to a keystore you already have"
    echo "   • Ideal if you're updating an existing configuration"
    echo
    echo "3. Create a new keystore with your certificate:"
    echo "   • Import your existing SSL certificate and private key into a keystore"
    echo
    echo "4. Use an existing keystore with certificate and private key:"
    echo "   • Use a keystore that already contains your certificate and private key"
    echo
    echo "Please enter 1, 2, 3, or 4 to select your preferred option:"
}

get_certificate_info() {
    if [ "$interactive" = true ]; then
        echo
        echo "Certificate Information Input"
        echo "-----------------------------"
        echo "You will now be guided through entering the certificate details manually."
        echo
        manual_input
    else
        echo
        echo "Certificate Information Input"
        echo "-----------------------------"
        echo "Using the ssl_config.properties file for certificate details."
        echo
        if [ -f "$SSL_CONFIG_FILE" ]; then
            read_properties_file
            # Validate properties based on the SSL option
            validate_properties "$ssl_option"
        else
            echo "Error: $SSL_CONFIG_FILE not found in the current directory."
            echo "In non-interactive mode, this file is required."
            exit 1
        fi
    fi
}

manual_input() {
    case $ssl_option in
        1)
            read -p "Domain name for this certificate: " domain_name
            read -p "Public IP of the server (optional): " server_public_ip
            read -p "Company name: " company_name
            read -p "Organizational unit (leave blank if none): " organizational_unit
            read -p "City/locality: " city
            read -p "State/province: " state
            read -p "Two-letter country code (e.g., US): " country_code
            read -p "Alias for the certificate in the keystore (dvsum-gateway-cert): " alias
            alias=${alias:-dvsum-gateway-cert}
            read -p "Keystore password: " keystore_pass
            ;;
        2)
            read -p "Path to the keystore: " keystore_path
            keystore_path=$(validate_file_path "$keystore_path" "keystore" "jks keystore ks")
            read -p "Alias for the certificate in the keystore (dvsum-gateway-cert): " alias
            alias=${alias:-dvsum-gateway-cert}
            read -p "Keystore password: " keystore_pass
            read -p "Path to server certificate file: " server_cert_path
            server_cert_path=$(validate_file_path "$server_cert_path" "server certificate" "crt pem cer")
            read -p "Path to root certificate file (leave blank if none): " root_cert_path
            if [ -n "$root_cert_path" ]; then
                root_cert_path=$(validate_file_path "$root_cert_path" "root certificate" "crt pem cer")
            fi
            read -p "Path to intermediate certificate file (leave blank if none): " intermediate_cert_path
            if [ -n "$intermediate_cert_path" ]; then
                intermediate_cert_path=$(validate_file_path "$intermediate_cert_path" "intermediate certificate" "crt pem cer")
            fi
            ;;
        3)
            read -p "Path to server certificate file: " server_cert_path
            server_cert_path=$(validate_file_path "$server_cert_path" "server certificate" "crt pem cer")
            read -p "Path to root certificate file (leave blank if none): " root_cert_path
            if [ -n "$root_cert_path" ]; then
                root_cert_path=$(validate_file_path "$root_cert_path" "root certificate" "crt pem cer")
            fi
            read -p "Path to intermediate certificate file (leave blank if none): " intermediate_cert_path
            if [ -n "$intermediate_cert_path" ]; then
                intermediate_cert_path=$(validate_file_path "$intermediate_cert_path" "intermediate certificate" "crt pem cer")
            fi
            read -p "Path to private key file: " private_key_path
            private_key_path=$(validate_file_path "$private_key_path" "private key" "key")
            read -p "Domain name for this certificate: " domain_name
            read -p "Company name: " company_name
            read -p "Organizational unit (leave blank if none): " organizational_unit
            read -p "City/locality: " city
            read -p "State/province: " state
            read -p "Two-letter country code (e.g., US): " country_code
            read -p "Alias for the certificate in the keystore (dvsum-gateway-cert): " alias
            alias=${alias:-dvsum-gateway-cert}
            read -p "Keystore password: " keystore_pass
            ;;
		4)
            read -p "Path to the existing keystore: " keystore_path
            keystore_path=$(validate_file_path "$keystore_path" "keystore" "keystore jks ks")
            read -p "Keystore password: " keystore_pass
            ;;
    esac
}

# --- SSL Configuration Functions ---

install_openssl() {
    if ! command -v openssl &> /dev/null; then
        echo "OpenSSL could not be found. Installing OpenSSL..."
        if command -v apt-get &> /dev/null; then
            sudo apt-get update && sudo apt-get install -y openssl
        elif command -v yum &> /dev/null; then
            sudo yum install -y openssl
        elif command -v dnf &> /dev/null; then
            sudo dnf install -y openssl
        else
            echo "Unable to install OpenSSL. Package manager not found. Please install OpenSSL manually."
            echo "Press Enter to exit..."
            read -r
            exit 1
        fi
    else
        echo "OpenSSL is already installed."
        echo 
    fi
}

create_keystore_and_csr() {
    keystore_path="$domain_name.p12"
    if [ -f "$keystore_path" ]; then
        rm "$keystore_path"
        echo "Existing $domain_name.p12 deleted."
    fi

    create_password_file

    # Set default key size if not specified
    if [ -z "$key_size" ]; then
        case "$encryption_algorithm" in
            RSA|DSA) key_size=2048 ;;
            EC) key_size=256 ;;
        esac
    fi

    $KEYTOOL \
        -genkeypair \
        -alias "$alias" \
        -keyalg "$encryption_algorithm" \
        -keysize "$key_size" \
        -keystore "$keystore_path" \
        -storepass:file "$temp_pass_file" \
        -dname "CN=$domain_name, OU=$organizational_unit, O=$company_name, L=$city, ST=$state, C=$country_code" \
        -ext "SAN=dns:localhost,dns:127.0.0.1${server_public_ip:+",ip:$server_public_ip"}" \
        -validity 3650
    if [ $? -ne 0 ]; then
        echo "Failed to create a new key pair. Exiting."
        process_successful=false
        remove_password_file
        return 1
    fi

    $KEYTOOL \
        -certreq \
        -alias "$alias" \
        -keystore "$keystore_path" \
        -storepass:file "$temp_pass_file" \
        -file "$domain_name.csr"
    if [ $? -ne 0 ]; then
        echo "Failed to create CSR. Exiting."
        process_successful=false
        remove_password_file
        return 1
    fi

    remove_password_file
}

convert_p12_to_jks() {
    jks_path="$domain_name.jks"
    if [ -f "$jks_path" ]; then
        rm "$jks_path"
        echo "Existing $domain_name.jks deleted."
    fi

    create_password_file

    $KEYTOOL \
        -importkeystore \
        -srckeystore "$keystore_path" \
        -srcstoretype PKCS12 \
        -destkeystore "$jks_path" \
        -deststoretype JKS \
        -srcstorepass:file "$temp_pass_file" \
        -deststorepass:file "$temp_pass_file"
    if [ $? -ne 0 ]; then
        echo "Failed to convert keystore. Exiting."
        process_successful=false
        remove_password_file
        return 1
    fi
    keystore_path="$domain_name.jks"
    remove_password_file
}

backup_keystore() {
    local backup_dir="ssl_tools/keystore_backups/$(date +%Y%m%d_%H%M%S)"
    
    # Create backup directory
    mkdir -p "$backup_dir"
    
    # Backup existing files if they exist
    local files_to_backup=("$domain_name.jks" "$domain_name.p12" "$domain_name.csr")
    local backup_performed=false
    
    for file in "${files_to_backup[@]}"; do
        if [ -f "$file" ]; then
            echo "Backing up existing $file"
            cp "$file" "$backup_dir/"
            backup_performed=true
        fi
    done
    
    if [ "$backup_performed" = true ]; then
        echo "Keystore files backed up to: $backup_dir"
    else
        rmdir "$backup_dir"
    fi
}

create_keystore_from_separate_files() {
    local new_keystore_path="$domain_name.jks"
    local temp_p12="$domain_name.p12"
    
    # Backup existing files
    backup_keystore
    
    # Delete existing .jks and .p12 files if they exist
    rm -f "$new_keystore_path" "$temp_p12"
    
    create_password_file

    # Create a PKCS12 keystore with the private key and server certificate
    openssl pkcs12 -export -in "$server_cert_path" -inkey "$private_key_path" -out "$temp_p12" -name "$alias" -passout file:"$temp_pass_file"
    if [ $? -ne 0 ]; then
        echo "Failed to create PKCS12 file. Exiting."
        process_successful=false
        remove_password_file
        return 1
    fi

    # Convert PKCS12 to JKS
    $KEYTOOL \
        -importkeystore \
        -srckeystore "$temp_p12" \
        -srcstoretype PKCS12 \
        -srcstorepass:file "$temp_pass_file" \
        -destkeystore "$new_keystore_path" \
        -deststoretype JKS \
        -deststorepass:file "$temp_pass_file" \
        -alias "$alias" \
        -noprompt
 
    if [ $? -ne 0 ]; then
        echo "Failed to create JKS keystore. Exiting."
        process_successful=false
        remove_password_file
        return 1
    fi

    # Import root and intermediate certificates if provided
    if [ -n "$root_cert_path" ]; then
        echo "Importing root certificate..."
        $KEYTOOL \
            -importcert \
            -alias "root" \
            -file "$root_cert_path" \
            -keystore "$new_keystore_path" \
            -storepass:file "$temp_pass_file" \
            -noprompt
        if [ $? -ne 0 ]; then
            echo "Failed to import root certificate."
            process_successful=false
        fi
    fi

    if [ -n "$intermediate_cert_path" ]; then
        echo "Importing intermediate certificate..."
        $KEYTOOL \
            -importcert \
            -alias "intermediate" \
            -file "$intermediate_cert_path" \
            -keystore "$new_keystore_path" \
            -storepass:file "$temp_pass_file" \
            -noprompt
        if [ $? -ne 0 ]; then
            echo "Failed to import intermediate certificate."
            process_successful=false
        fi
    fi

    remove_password_file

    if [ "$process_successful" = true ]; then
        echo "New keystore created successfully with imported certificate and private key for domain: $domain_name"
        keystore_path="$new_keystore_path"
		
    else
        echo "Keystore creation failed. View error logs"
    fi
}

import_certificates() {
    create_password_file
	
	# Backup existing files
    backup_keystore

    import_cert() {
        local alias=$1
        local cert_path=$2
        local is_server_cert=$3

        if [ "$is_server_cert" = true ]; then
            # For server certificate, we're updating the existing entry
            $KEYTOOL \
                -importcert \
                -alias "$alias" \
                -file "$cert_path" \
                -keystore "$keystore_path" \
                -storepass:file "$temp_pass_file" \
                -trustcacerts \
                -noprompt
        else
            # For root and intermediate certs, we're adding new entries
            $KEYTOOL \
                -importcert \
                -alias "$alias" \
                -file "$cert_path" \
                -keystore "$keystore_path" \
                -storepass:file "$temp_pass_file" \
                -trustcacerts \
                -noprompt
        fi

        if [ $? -ne 0 ]; then
            echo "Failed to import certificate with alias '$alias'."
            process_successful=false
        else
            echo "Certificate imported successfully with alias '$alias'."
        fi
    }

    if [ -n "$root_cert_path" ]; then
        import_cert "root" "$root_cert_path" false
    fi

    if [ -n "$intermediate_cert_path" ]; then
        import_cert "intermediate" "$intermediate_cert_path" false
    fi

    # For server certificate alias
    import_cert "$alias" "$server_cert_path" true

    remove_password_file

    if [ "$process_successful" = true ]; then
        echo "All certificates imported successfully."
    else
        echo "Certificate import process completed with errors."
    fi
}

import_additional_certificates() {
    if [ -n "$root_cert_path" ]; then
        echo "Adding root certificate ......."
        $KEYTOOL \
            -importcert \
            -alias root \
            -file "$root_cert_path" \
            -keystore "$new_keystore_path" \
            -storepass:file "$temp_pass_file" \
            -noprompt
        if [ $? -ne 0 ]; then
            echo "Failed to import root certificate."
        fi
    fi

    if [ -n "$intermediate_cert_path" ]; then
        echo "Adding intermediate certificate ............."
        $KEYTOOL \
            -importcert \
            -alias chain \
            -file "$intermediate_cert_path" \
            -keystore "$new_keystore_path" \
            -storepass:file "$temp_pass_file" \
            -noprompt
        if [ $? -ne 0 ]; then
            echo "Failed to import intermediate certificate."
        fi
    fi
}

validate_yes_no_input() {
    local prompt="$1"
    local response
    while true; do
        read -p "$prompt (y/n): " response
        case "${response,,}" in
            y|yes|n|no)
                echo "${response,,}"
                return
                ;;
            *)
                echo "Invalid input. Please enter 'y' for yes or 'n' for no."
                ;;
        esac
    done
}

verify_keystore() {
    if [ "$process_successful" = true ]; then
        echo
        if [ "$interactive" = true ]; then
            read -p "Do you want to verify the keystore contents? (y/n): " verify_choice
            if [ "${verify_choice,,}" = "y" ]; then
                perform_keystore_verification
            else
                echo "Skipping keystore verification."
            fi
        fi

        echo
        echo "SSL Configuration Properties for your Java application:"
        echo "key-store=$keystore_path"
        echo "key-store-password=$keystore_pass"
        echo "key-store-type=JKS"
        echo "key-alias=$alias"

        echo
        echo "SSL Configuration completed successfully. Your keystore is ready for use in your Java application."
        echo "Please securely store your keystore file and password."
    else
        echo
        echo "SSL Configuration process completed with errors. Please review the output above for details."
        if [ "$interactive" = true ]; then
            echo "Press Enter to continue..."
            read -r  # This waits for the user to press Enter
        fi
    fi
}

perform_keystore_verification() {

    # Check if keystore_path and keystore_pass are set
    if [ -z "$keystore_path" ] || [ -z "$keystore_pass" ]; then
        echo "Keystore path or password not set. Attempting to read from ssl_config.properties..."
        if [ -f "$SSL_CONFIG_FILE" ]; then
            while IFS='=' read -r key value; do
                key=$(echo "$key" | sed 's/^[ \t]*//;s/[ \t]*$//')
                value=$(echo "$value" | sed 's/^[ \t]*//;s/[ \t]*$//')
                if [ -n "$key" ] && [[ ! "$key" =~ ^# ]]; then
                    case "$key" in
                        keystore_path)
                            keystore_path="$value"
                            ;;
                        keystore_pass)
                            keystore_pass="$value"
                            ;;
                    esac
                fi
            done < "$SSL_CONFIG_FILE"
        else
            echo "Error: $SSL_CONFIG_FILE not found."
            return 1
        fi
    fi

    if [ -z "$keystore_path" ] || [ -z "$keystore_pass" ]; then
        echo "Error: Unable to determine keystore path or password."
        return 1
    fi

    # Verify the keystore
    echo
    echo "Verifying keystore contents:"
    create_password_file
    $KEYTOOL -list -v -keystore "$keystore_path" -storepass:file "$temp_pass_file"
    remove_password_file
    if [ $? -ne 0 ]; then
        echo "Error: Failed to verify keystore."
        return 1
    fi
}

generate_ssl_config_file() {
    # Read existing properties from the file if it exists
    if [ -f "$SSL_CONFIG_FILE" ]; then
        while IFS='=' read -r key value; do
            # Remove any leading/trailing whitespace from key and value
            key=$(echo "$key" | sed 's/^[ \t]*//;s/[ \t]*$//')
            value=$(echo "$value" | sed 's/^[ \t]*//;s/[ \t]*$//')

            # Skip empty lines and comments
            if [ -n "$key" ] && [[ ! "$key" =~ ^# ]]; then
                # Correctly handle values with spaces or special characters
                # No need to add additional quotes here, just store the value as is
                eval "existing_${key}='$value'"
            fi
        done < "$SSL_CONFIG_FILE"
    fi

    # Function to get existing or new value
    get_value() {
        local key=$1
        local -n existing="existing_${key}"

        # Output the value directly without adding extra quotes
        if [[ -n "$2" ]]; then
            echo "$2"
        else
            echo "${existing}"
        fi
    }

    # Build the configuration content with potential overwrite
    config_content=$(mktemp)
    {
        echo "# SSL Configuration Properties"
        echo "# This file contains the necessary configurations for SSL setup and DvSum Gateway service."
        echo ""
        echo "# ----------------------------------------"
        echo "# 1. Domain and Company Information"
        echo "# ----------------------------------------"
        echo "# These settings are used for certificate generation and identification"
        echo "domain_name=$(get_value domain_name "$domain_name")"
        echo "company_name=$(get_value company_name "$company_name")"
        echo "organizational_unit=$(get_value organizational_unit "$organizational_unit")"
        echo "city=$(get_value city "$city")"
        echo "state=$(get_value state "$state")"
        echo "country_code=$(get_value country_code "$country_code")"
        echo "# Optional: Public IP of the server (leave blank if not applicable)"
        echo "server_public_ip=$(get_value server_public_ip "$server_public_ip")"
        echo ""
        echo "# ----------------------------------------"
        echo "# 2. Keystore Configuration"
        echo "# ----------------------------------------"
        echo "# Settings for the Java keystore that will hold the SSL certificates"
        echo "keystore_path=$(get_value keystore_path "$keystore_path")"
        echo "keystore_pass=$(get_value keystore_pass "$keystore_pass")"
        echo "key_store_type=JKS"
        echo "alias=$(get_value alias "$alias")"
        echo ""
        echo "# ----------------------------------------"
        echo "# 3. Certificate Paths"
        echo "# ----------------------------------------"
        echo "# Paths to your SSL certificate files"
        echo "# Note: root_cert_path and intermediate_cert_path are optional"
        echo "server_cert_path=$(get_value server_cert_path "$server_cert_path")"
        echo "private_key_path=$(get_value private_key_path "$private_key_path")"
        echo "root_cert_path=$(get_value root_cert_path "$root_cert_path")"
        echo "# Leave blank if no intermediate certificate"
        echo "intermediate_cert_path=$(get_value intermediate_cert_path "$intermediate_cert_path")"
        echo ""
        echo "# ----------------------------------------"
        echo "# 4. Additional Configuration (if needed)"
        echo "# ----------------------------------------"
        echo "# Add any additional configuration parameters here"
    } > "$config_content"

    # Compare the new content with the existing file
    if [ -f "$SSL_CONFIG_FILE" ] && cmp -s "$config_content" "$SSL_CONFIG_FILE"; then
        echo "No changes detected. The configuration file is up-to-date."
        rm "$config_content"
        return
    fi

    mv "$config_content" "$SSL_CONFIG_FILE"
    echo "Configuration has been updated with the latest values."
    echo "The updated configuration file is: $SSL_CONFIG_FILE"
}



# --- Service Management Functions ---

create_service_file() {
    local service_file="${1:?Service file path is required}"
    local jar_file="${2:?JAR file path is required}"
    local description="${3:?Service description is required}"
	local service_name="${4:?Service name is required}"

    cat <<EOF | sudo tee "$service_file" > /dev/null
[Unit]
Description=$description
After=network.target

[Service]
Type=simple
WorkingDirectory=$(dirname "$jar_file")
ExecStart=$JRE_DIR/bin/java -jar $jar_file
User=$(whoami)
Restart=on-failure

[Install]
WantedBy=multi-user.target
EOF

    sudo chmod 644 "$service_file"
}

create_python_service_file() {
    local service_file="${1:?Service file path is required}"
    local flask_app="${2:?Flask app path is required}"
    local description="${3:?Service description is required}"
    local service_name="${4:?Service name is required}"

    cat <<EOF | sudo tee "$service_file" > /dev/null
[Unit]
Description=$description
After=network.target

[Service]
Type=simple
WorkingDirectory=$(dirname "$flask_app")
ExecStart=/usr/bin/python3 $flask_app
User=$(whoami)
Restart=on-failure
Environment="PYTHONUNBUFFERED=1"  # Ensures Python output is immediately flushed (good for logging)
Environment="FLASK_ENV=production"  # Set Flask environment to production
Environment="FLASK_APP=$flask_app"
Environment="FLASK_RUN_PORT=$python_service_port_number"  # Use the port defined in your variables

[Install]
WantedBy=multi-user.target
EOF

    sudo chmod 644 "$service_file"
}


manage_services() {
    local action=$1
    for service in "$gateway_service_name" "$gateway_monitor_name" "$gateway_python_service_name"; do
        if ! systemctl list-unit-files | grep -q "$service.service"; then
            echo "Error: $service service does not exist."
            continue
        fi

        case $action in
            start|restart)
                sudo systemctl $action "$service.service"
                if [ $? -eq 0 ]; then
                    echo "$service service has been ${action}ed."
                else
                    echo "Failed to $action $service service. Check system logs for more information."
                fi
                ;;
            stop)
                if systemctl is-active --quiet "$service.service"; then
                    sudo systemctl stop "$service.service"
                    if [ $? -eq 0 ]; then
                        echo "$service service has been stopped."
                    else
                        echo "Failed to stop $service service. Check system logs for more information."
                    fi
                else
                    echo "$service service is not running."
                fi
                ;;
            status)
                if systemctl is-active --quiet "$service.service"; then
                    echo "$service service is running."
                    sudo systemctl status "$service.service" | grep -E "Active:|Main PID:"
                else
                    echo "$service service is not running."
                fi
                ;;
        esac
        echo "----------------------------------------"
    done
}


uninstall_services() {
    echo "Uninstalling DvSum Gateway and Python services..."
    
    for service in "$gateway_service_name" "$gateway_monitor_name" "$gateway_python_service_name"; do
        if ! systemctl list-unit-files | grep -q "$service.service"; then
            echo "${service}.service does not exist."
        else
            if systemctl is-active --quiet "${service}.service"; then
                sudo systemctl stop "${service}.service"
            fi
            # Disable the service
            sudo systemctl disable "${service}.service"
            # Remove the service file
            sudo rm "/etc/systemd/system/${service}.service"
            echo "${service} service has been uninstalled."
        fi
    done
    
    # Reload the systemd daemon to apply the changes
    sudo systemctl daemon-reload
}


install_pip() {
    echo "Pip is not installed. Attempting to install pip..."

    # Download get-pip.py
    curl -O https://bootstrap.pypa.io/get-pip.py

    # Run the Python script to install pip
    if command -v python3 >/dev/null 2>&1; then
        python3 get-pip.py
    elif command -v python >/dev/null 2>&1; then
        python get-pip.py
    else
        echo "Python is not installed. Please install Python first."
        exit 1
    fi

    # Verify pip installation
    if command -v pip3 >/dev/null 2>&1 || command -v pip >/dev/null 2>&1; then
        echo "Pip successfully installed."
    else
        echo "Failed to install pip. Exiting."
        exit 1
    fi

    # Clean up get-pip.py file
    rm -f get-pip.py
}

install_python_dependencies() {
    echo "Installing Python dependencies from ${REQUIREMENTS_FILE}..."
    
    if [ -f "$REQUIREMENTS_FILE" ]; then
        # Check for pip3 first, then pip, and install if neither is found
        if command -v pip3 >/dev/null 2>&1; then
            PIP_COMMAND="pip3"
        elif command -v pip >/dev/null 2>&1; then
            PIP_COMMAND="pip"
        else
            # Pip is not installed, so try to install it
            install_pip

            # Set PIP_COMMAND after pip installation
            if command -v pip3 >/dev/null 2>&1; then
                PIP_COMMAND="pip3"
            elif command -v pip >/dev/null 2>&1; then
                PIP_COMMAND="pip"
            else
                echo "Pip installation failed. Exiting."
                exit 1
            fi
        fi

        # Install dependencies using the detected pip command
        $PIP_COMMAND install -r "$REQUIREMENTS_FILE"
        
        # Check if the pip install succeeded
        if [ $? -ne 0 ]; then
            echo "Failed to install Python dependencies from ${REQUIREMENTS_FILE}. Exiting."
            exit 1
        else
            echo "Python dependencies installed successfully."
        fi
    else
        echo "Requirements file ${REQUIREMENTS_FILE} not found. Exiting."
        exit 1
    fi
}



setup_services() {
    echo 
    echo "Setting up DvSum Gateway and Python Flask services..."
    echo "------------------------------------"

    for service in "$gateway_service_name" "$gateway_monitor_name" "$gateway_python_service_name"; do
        service_file="/etc/systemd/system/${service}.service"
        
        if systemctl is-active --quiet "${service}.service"; then
            echo "Restarting ${service} service..."
            sudo systemctl restart "${service}.service"
            echo "${service} service has been restarted."
        else
            if [ -f "$service_file" ]; then
                echo "${service} service file exists but the service is not running. Starting the service..."
                sudo systemctl start "${service}.service"
            else
                echo "Installing ${service} service..."
                
                # Install Python dependencies for the Python service
                if [ "$service" = "$gateway_python_service_name" ]; then
                    install_python_dependencies  # Call the new function to install dependencies
                    # Create the Python Flask service
                    create_python_service_file "$service_file" "$FLASK_APP" "DvSum Python Flask Service" "$service"
                elif [ "$service" = "$gateway_service_name" ]; then
                    # Create the Java Gateway service
                    create_service_file "$service_file" "$DVSUM_JAR" "DvSum Gateway daemon" "$service"
                elif [ "$service" = "$gateway_monitor_name" ]; then
                    # Create the Java Monitor service
                    create_service_file "$service_file" "$DVSUM_MONITOR_JAR" "DvSum Gateway monitor daemon" "$service"
                fi
                
                # Reload systemd daemon to recognize the new service
                sudo systemctl daemon-reload
                # Enable and start the service
                sudo systemctl enable "${service}.service"
                sudo systemctl start "${service}.service"
            fi
            echo "${service} service has been installed and started."
        fi
        
        # Display service status
        sudo systemctl status "${service}.service"
    done
}


# --- Main Execution ---

main() {
    parse_arguments "$@"

    case "$action" in
        start|restart)
            restart_services
            exit 0
            ;;
        stop)
            stop_services
            exit 0
            ;;
        uninstall)
            uninstall_services
            exit 0
            ;;
        status)
            check_service_status
            exit 0
            ;;
        install)
            perform_installation
            exit 0
            ;;
        "")
            perform_installation
            exit 0
            ;;
        *)
            echo "Error: Unknown action '$action'"
            display_usage
            exit 1
            ;;
    esac
}
validate_encryption_algorithm() {
    local algo="$1"
    local valid_algos=("RSA" "DSA" "EC")
    for valid_algo in "${valid_algos[@]}"; do
        if [ "$algo" = "$valid_algo" ]; then
            return 0
        fi
    done
    echo "Invalid encryption algorithm. Valid options are: ${valid_algos[*]}"
    return 1
}

validate_key_size() {
    local algo="$1"
    local size="$2"
    case "$algo" in
        RSA)
            if [[ "$size" =~ ^(2048|3072|4096)$ ]]; then
                return 0
            fi
            echo "For RSA, valid key sizes are 2048, 3072, or 4096 bits."
            ;;
        DSA)
            if [[ "$size" =~ ^(1024|2048|3072)$ ]]; then
                return 0
            fi
            echo "For DSA, valid key sizes are 1024, 2048, or 3072 bits."
            ;;
        EC)
            if [[ "$size" =~ ^(256|384|521)$ ]]; then
                return 0
            fi
            echo "For EC, valid key sizes are 256, 384, or 521 bits."
            ;;
    esac
    return 1
}

perform_installation() {
    # SSL Configuration
    if [ "$interactive" = false ]; then
        if [ -z "$ssl_option" ]; then
            echo "Error: In non-interactive mode, --ssl-option must be provided. Example:"
            echo "dvsum_gateway_setup.sh --ssl-option=<option> --non-interactive"
            echo "Replace <option> with 1, 2, 3 or 4 based on your SSL configuration needs:"
            echo " 1. For Generating a new CSR or Self-signed certificate"
            echo " 2. For adding certificates into an existing keystore file"
            echo " 3. For Importing server certificate and private key into a new keystore"
			echo " 4. For using an existing keystore already containing the server certificate and private key"
            exit 1
        fi
        read_properties_file
		check_network_connectivity
    else
        display_welcome_message
        install_openssl
		get_gateway_config
        check_network_connectivity
        display_ssl_menu
        read -p "Enter your choice (1, 2, 3 or 4): " ssl_option
    fi

    # Validate ssl_option
    if [[ ! "$ssl_option" =~ ^[1-4]$ ]]; then
        echo "Error: Invalid SSL option. Must be 1, 2, 3, or 4."
        exit 1
    fi

    # Validate encryption algorithm and key size
    if ! validate_encryption_algorithm "$encryption_algorithm"; then
        exit 1
    fi
    if [ -n "$key_size" ] && ! validate_key_size "$encryption_algorithm" "$key_size"; then
        exit 1
    fi

    case $ssl_option in
        1)
            if [ "$interactive" = true ]; then
                get_certificate_info
            fi
            keystore_path="$domain_name.p12"
            create_keystore_and_csr
            convert_p12_to_jks
            if [ "$process_successful" = true ]; then
                echo "CSR created successfully. The CSR file is saved as $domain_name.csr."
                echo "Keystore converted to JKS format ($domain_name.jks)"
            fi
            ;;
        2)
            if [ "$interactive" = true ]; then
                get_certificate_info
            fi
            keystore_path=$(validate_file_path "$keystore_path" "keystore" "jks keystore ks")
            import_certificates
            verify_keystore
            ;;
        3)
            if [ "$interactive" = true ]; then
                get_certificate_info
            fi
            create_keystore_from_separate_files
            verify_keystore
            ;;
        4)
            if [ "$interactive" = true ]; then
                get_certificate_info
            fi
            verify_keystore
            if [ "$process_successful" = false ]; then
                echo "Failed to verify existing keystore. Exiting."
                exit 1
            fi
            ;;
    esac

    generate_ssl_config_file
    setup_services

    display_completion_message
}

parse_arguments() {
    # Check for help request
    if [[ "$1" == "--help" || "$1" == "help" || "$1" == "-help" ]]; then
        display_help
        exit 0
    fi
	
    # If no arguments are provided, default to install
    if [ $# -eq 0 ]; then
        action="install"
        return
    fi

    while [[ $# -gt 0 ]]; do
        case "$1" in
			--gateway-service-name=*)
                gateway_service_name="${1#*=}"
                shift
                ;;
            --gateway-monitor-name=*)
                gateway_monitor_name="${1#*=}"
                shift
                ;;
            --gateway-python-service-name=*)
                gateway_python_service_name="${1#*=}"
                shift
                ;;
            --non-interactive)
                interactive=false
                shift
                ;;
            --ssl-option=*)
                ssl_option="${1#*=}"
                shift
                ;;
            --encryption-algorithm=*)
                encryption_algorithm="${1#*=}"
                shift
                ;;
            --key-size=*)
                key_size="${1#*=}"
                shift
                ;;
			--check-network)
				check_network_connectivity
                exit $?
                ;;
            --verify-keystore)
                perform_keystore_verification
                exit $?
                ;;
            start|restart|stop|uninstall|status|install)
                if [ -n "$action" ]; then
                    echo "Error: Only one action can be specified."
                    display_usage
                    exit 1
                fi
                action="$1"
                shift
                ;;
            *)
                echo "Error: Unknown option or action '$1'"
                display_usage
                exit 1
                ;;
        esac
    done

    # If no action was specified in the arguments, default to install
    if [ -z "$action" ]; then
        action="install"
    fi
}

main() {
    parse_arguments "$@"

    case "$action" in
        start|restart|stop)
            # Handle proper verb forms for action
            case "$action" in
                start) action_display="Starting" ;;
                restart) action_display="Restarting" ;;
                stop) action_display="Stopping" ;;
            esac
            echo "$action_display DvSum Gateway services..."
            manage_services "$action"
            ;;
        uninstall)
            uninstall_services
            ;;
        status)
            manage_services status
            ;;
        install)
            perform_installation
            ;;
        *)
            echo "Error: Invalid action '$action'"
            display_usage
            exit 1
            ;;
    esac
}


display_usage() {
    echo
    echo "Usage: dvsum_gateway_setup.sh [OPTIONS] [ACTION]"
    echo
    echo "OPTIONS:"
    echo "  --gateway-service-name=<name>         Specify custom name for the gateway service"
    echo "  --gateway-monitor-name=<name>         Specify custom name for the monitor service"
    echo "  --gateway-python-service-name=<name>  Specify custom name for the python service"
    echo "  --non-interactive                     Run in non-interactive mode"
    echo "  --ssl-option=<1|2|3|4>                Specify SSL configuration option"
    echo "  --encryption-algorithm=<algo>         Specify encryption algorithm (RSA, DSA, EC)"
    echo "  --key-size=<size>                     Specify key size (depends on algorithm)"
    echo "  --verify-keystore                     Verify the contents of the keystore"
    echo "  --check-network                       Verify Outbound Access to DvSum Web Application"
    echo "  --help                                Display the help for using this script"
    echo
    echo "ACTIONS:"
    echo "  start                   Start the DvSum Gateway services"
    echo "  stop                    Stop the DvSum Gateway services"
    echo "  restart                 Restart the DvSum Gateway services"
    echo "  status                  Check the status of the DvSum Gateway services"
    echo "  uninstall               Uninstall the DvSum Gateway services"
    echo "  install                 Perform full installation and configuration (default)"
    echo
    echo "Examples:"
    echo "  dvsum_gateway_setup.sh --gateway-service-name=CustomGateway --gateway-monitor-name=CustomMonitor --gateway-python-service-name=CustomPythonService install"
    echo "  dvsum_gateway_setup.sh --ssl-option=1 --non-interactive"
    echo "  dvsum_gateway_setup.sh --ssl-option=2 --encryption-algorithm=RSA --key-size=2048 install"
    echo "  dvsum_gateway_setup.sh start"
    echo "  dvsum_gateway_setup.sh stop"
    echo "  dvsum_gateway_setup.sh status"
    echo
    exit 0
}


display_help() {
	echo  
    echo "DvSum Gateway Manager"
    echo "----------------------"
    echo
    echo "This script manages the DvSum Gateway, including installation, SSL configuration,"
    echo "and service control."
    echo
    display_usage
}

display_failure_message() {
    echo
    echo "❌ DvSum Gateway Installation and Configuration Failed"
    echo "-----------------------------------------------------"
    echo "We encountered issues during the setup process."
    echo "Please review the error messages above for details on what went wrong."
    echo
    echo "For assistance, please contact DvSum support at support@dvsum.com"
    echo
    if [ "$interactive" = true ]; then
        read -p "Press Enter to exit..."
    fi
}

display_success_message(){
	# Completion message
	echo
	echo "✅ DvSum Gateway Installation and Configuration Complete!"
	echo "--------------------------------------------------------"
	echo "Great news! The setup process has finished successfully."
	echo
	echo "Here's what we've accomplished:"
	echo "1. SSL has been configured for secure communication"
	echo "2. DvSum Gateway services have been installed"
	echo "3. All necessary services have been started"
	echo
	echo "Next steps:"
	echo "• Verify that the Gateway is accessible at: https://<your-domain>:<port>"
	echo "• Check the logs at ./logs/SAWS.log for any important messages"
	echo "• Refer to the DvSum documentation for post-installation tasks"
	echo
	echo "If you encounter any issues or have questions, please contact"
	echo "DvSum support at support@dvsum.com"
	echo
	echo "Thank you for choosing DvSum Gateway!"
	echo
    if [ "$interactive" = true ]; then
        read -p "Press enter to complete the installation..."
    fi
}
display_completion_message(){
	if [ "$process_successful" = true ]; then
		display_success_message
	else
		display_failure_message
	fi
}


# Call the main function
main "$@"