@echo off
setlocal enabledelayedexpansion

REM Set global variables
set "JRE_DIR=%cd%\jre1.8.0_202"
set "DVSUM_JAR=%cd%\DvSum.jar"
set "DVSUM_MONITOR_JAR=%cd%\DvSum_monitor.jar"
set "SSL_CONFIG_FILE=ssl_config.properties"
set "KEYTOOL=%cd%\ssl_tools\openlogic-jre-21.0.3+9\bin\keytool.exe"
set "OPENSSL=%cd%\ssl_tools\openssl-3\x64\bin\openssl.exe"
set "FLASK_APP=%cd%\python_service\flask_app.py"
set "REQUIREMENTS_FILE=%cd%\python_service\requirements.txt"


REM Set Java file encoding
set "JAVA_TOOL_OPTIONS=-Dfile.encoding=UTF-8"

REM Set global flags
set "process_successful=true"
set "interactive=true"
set "ssl_option="
set "action="
set "encryption_algorithm=RSA"
set "key_size=2048"

REM Parse command-line arguments
:parse_args
if "%~1"=="" goto :end_parse_args
set "arg=%~1"
if "%arg:~0,2%"=="--" (
    for /f "tokens=1,* delims==" %%a in ("%arg%") do (
        set "option=%%a"
        set "value=%%b"
        if "!value!"=="" (
            if "%~2"=="" (
                if "!option!"=="--non-interactive" (
                    set "interactive=false"
                ) else if "!option!"=="--check-network" (
                    set "action=check_network"
                ) else if "!option!"=="--verify-keystore" (
                    set "action=check_keystore_content"
                ) else (
                    call :display_usage
                    exit /b 1
                )

            ) else (
                set "value=%~2"
                shift
            )
        )
        if "!option!"=="--gateway-service-name" set "gateway_service_name=!value!"
        if "!option!"=="--gateway-monitor-name" set "gateway_monitor_name=!value!"
        if "!option!"=="--gateway-python-service-name" set "gateway_python_service_name=!value!"
        if "!option!"=="--ssl-option" set "ssl_option=!value!"
        if "!option!"=="--encryption-algorithm" set "encryption_algorithm=!value!"
        if "!option!"=="--key-size" set "key_size=!value!"
        if "!option!"=="--non-interactive" set "interactive=false"
		if "!option!"=="--api-key" set "cmd_api_key=!value!"
		if "!option!"=="--port" set "cmd_port=!value!"
    )
) else (
    if /i "%arg%"=="start" set "action=start"
    if /i "%arg%"=="stop" set "action=stop"
    if /i "%arg%"=="restart" set "action=restart"
    if /i "%arg%"=="status" set "action=status"
    if /i "%arg%"=="uninstall" set "action=uninstall"
    if /i "%arg%"=="install" set "action=install"
	if /i "%arg%"=="check-network" set "action=check_network"
    if /i "%arg%"=="--help" set "action=help"
	if /i "%arg%"=="--verify-keystore" set "action=check_keystore_content"
    if /i "%arg%"=="help" set "action=help"
    if /i "%arg%"=="-help" set "action=help"
    if not defined action (
        echo Unknown option: %arg%
        call :display_usage
        exit /b 1
    )
)
shift
goto :parse_args
:end_parse_args

if not defined action set "action=install"
if not defined gateway_service_name set "gateway_service_name=DvSum-Gateway"
if not defined gateway_monitor_name set "gateway_monitor_name=DvSum-Gateway-Monitor"
if not defined gateway_python_service_name set "gateway_python_service_name=DvSum-Python"

REM Main execution
call :main
exit /b %ERRORLEVEL%

:main
if "%action%"=="" set "action=install"

REM Handle different actions
if "%action%"=="start" (
    call :start_services
    exit /b %ERRORLEVEL%
) else if "%action%"=="stop" (
    call :stop_services
    exit /b %ERRORLEVEL%
) else if "%action%"=="restart" (
    call :restart_services
    exit /b %ERRORLEVEL%
) else if "%action%"=="status" (
    call :check_service_status
    exit /b %ERRORLEVEL%
) else if "%action%"=="uninstall" (
    call :uninstall_services
    exit /b %ERRORLEVEL%
) else if "%action%"=="install" (
    call :perform_installation
    exit /b %ERRORLEVEL%
) else if "%action%"=="help" (
    call :display_help
    exit /b %ERRORLEVEL%
) else if "%action%"=="check_network" (
    call :check_network_connectivity
    exit /b %ERRORLEVEL%
) else if "%action%"=="check_keystore_content" (
	call :read_properties_file
    call :check_keystore_content
    exit /b %ERRORLEVEL%
)  else (
    echo Error: Invalid action '%action%'
    call :display_usage
    exit /b 1
)

:check_network_connectivity
setlocal EnableDelayedExpansion
echo.
echo Checking network connectivity to DvSum application ...
echo -------------------------------------------------------

set "output_file=%cd%\network_health_check_result.txt"

REM Call the Java method to perform checks and write output file
REM Redirect both stdout and stderr to NUL to suppress all output
"%JRE_DIR%\bin\java" -cp DvSum.jar com.dvsum.services.NetworkHealthChecker "%output_file%" >NUL 2>&1

if not exist "%output_file%" (
    echo Error: Output file was not created by the network health check.
    exit /b 1
)

echo.

REM Display the formatted output
type "%output_file%"

echo.
echo -------------------------------------------------------

REM Clean up the temporary file
del "%output_file%" 2>nul

endlocal
exit /b 0


:perform_installation
if "%interactive%"=="false" (
    if "%ssl_option%"=="" (
        echo Error: In non-interactive mode, --ssl-option must be provided.
        call :display_usage
        exit /b 1
    )
    call :read_properties_file
	call :check_network_connectivity 
) else (
    call :display_welcome_message
	call :update_configurations
	call :check_network_connectivity 
    call :display_ssl_menu
    set /p ssl_option="Enter your choice (1, 2, 3, 4): "
)

call :validate_ssl_option
if !errorlevel! neq 0 exit /b 1

if "%ssl_option%"=="1" (
    if "%interactive%"=="true" call :get_certificate_info
    set "keystore_path=%domain_name%.p12"
    call :create_keystore_and_csr
    call :convert_p12_to_jks
) else if "%ssl_option%"=="2" (
    if "%interactive%"=="true" call :get_certificate_info
    call :import_certificates
    call :verify_keystore
) else if "%ssl_option%"=="3" (
    if "%interactive%"=="true" call :get_certificate_info
    set "new_keystore_path=%domain_name%.jks"
    call :create_keystore_from_separate_files
    call :verify_keystore
) else if "%ssl_option%"=="4" (
    if "%interactive%"=="true" call :get_certificate_info
    call :verify_existing_keystore
    if !errorlevel! neq 0 exit /b 1
)

call :generate_ssl_config_file
call :setup_services
call :setup_python_service
call :display_completion_message
exit /b 0

:display_welcome_message
echo.
echo Welcome to the DvSum Gateway Installation and Configuration Wizard!
echo -------------------------------------------------------------------
echo This wizard will guide you through the process of:
echo   1. Setting up SSL for secure communication
echo   2. Installing and configuring the DvSum Gateway as a service on your Windows machine
echo.
echo You'll be asked to provide information for:
echo   . SSL certificate details
echo   . Keystore configuration
echo   . Service setup preferences
echo.
echo Please ensure you have the necessary SSL certificates and server information ready.
echo If you're unsure about any step, please consult the DvSum Gateway documentation.
echo.
echo Let's get started!
echo.
exit /b 0

:display_ssl_menu
echo SSL Configuration Options:
echo -------------------------
echo 1. Generate a new certificate:
echo    . Create a Certificate Signing Request (CSR) for a CA-signed certificate
echo    . Or generate a self-signed certificate for testing
echo.
echo 2. Add certificates to an existing keystore:
echo    . Add your SSL certificate to a keystore you already have
echo    . Ideal if you're updating an existing configuration
echo.
echo 3. Create a new keystore with your certificate:
echo    . Import your existing SSL certificate and private key into a keystore
echo.
echo 4. Use an existing keystore with certificate and private key:
echo    . Use a keystore that already contains your certificate and private key
echo.
echo Please enter 1, 2, 3, or 4 to select your preferred option:
exit /b 0

:get_certificate_info
if "%interactive%"=="true" (
    echo.
    echo Certificate Information Input
    echo -----------------------------
    echo You will now be guided through entering the certificate details manually.
    echo.
    call :manual_input
) else (
    echo.
    echo Certificate Information Input
    echo -----------------------------
    echo Using the %SSL_CONFIG_FILE% file for certificate details.
    echo.
    if exist "%SSL_CONFIG_FILE%" (
        call :read_properties_file
    ) else (
        echo Error: %SSL_CONFIG_FILE% not found in the current directory.
        echo In non-interactive mode, this file is required.
        exit /b 1
    )
)
exit /b 0

:read_properties_file
if not exist "%SSL_CONFIG_FILE%" (
    echo Error: Properties file not found: %SSL_CONFIG_FILE%
    exit /b 1
)
for /f "tokens=1,* delims==" %%a in (%SSL_CONFIG_FILE%) do (
    set "%%a=%%b"
)
call :validate_properties
exit /b 0

:validate_properties
set "missing_props="
if "%ssl_option%"=="1" (
    for %%i in (domain_name company_name organizational_unit city state country_code alias keystore_pass) do (
        if not defined %%i set "missing_props=!missing_props! %%i"
    )
) else if "%ssl_option%"=="2" (
    for %%i in (keystore_path alias keystore_pass server_cert_path) do (
        if not defined %%i set "missing_props=!missing_props! %%i"
    )
) else if "%ssl_option%"=="3" (
    for %%i in (server_cert_path private_key_path domain_name company_name organizational_unit city state country_code alias keystore_pass) do (
        if not defined %%i set "missing_props=!missing_props! %%i"
    )
) else if "%ssl_option%"=="4" (
    for %%i in (keystore_path keystore_pass) do (
        if not defined %%i set "missing_props=!missing_props! %%i"
    )
)
if defined missing_props (
    echo Error: The following required properties are missing in ssl_config.properties for SSL option %ssl_option%:
    echo %missing_props%
    endlocal
    exit 1
)
exit /b 0

:manual_input
setlocal enabledelayedexpansion

if "%ssl_option%"=="1" (
    call :get_input domain_name "Domain name for this certificate"
    set /p server_public_ip="Public IP of the server (optional): "
    call :get_input company_name "Company name"
    set /p organizational_unit="Organizational unit (leave blank if none): "
    call :get_input city "City/locality"
    call :get_input state "State/province"
    call :get_input country_code "Two-letter country code (e.g., US)"
    call :get_input alias "Alias for the certificate in the keystore" "dvsum-gateway-cert"
    call :get_input keystore_pass "Keystore password"
) else if "%ssl_option%"=="2" (
    call :get_input keystore_path "Path to the keystore"
    call :validate_file_path keystore_path "keystore" "jks keystore ks"
    call :get_input alias "Alias for the certificate in the keystore" "dvsum-gateway-cert"
    call :get_input keystore_pass "Keystore password"
    call :get_input server_cert_path "Path to server certificate file"
    call :validate_file_path server_cert_path "server certificate" "crt pem cer"
    set /p root_cert_path="Path to root certificate file (leave blank if none): "
    if defined root_cert_path call :validate_file_path root_cert_path "root certificate" "crt pem cer"
    set /p intermediate_cert_path="Path to intermediate certificate file (leave blank if none): "
    if defined intermediate_cert_path call :validate_file_path intermediate_cert_path "intermediate certificate" "crt pem cer"
) else if "%ssl_option%"=="3" (
    call :get_input server_cert_path "Path to server certificate file"
    call :validate_file_path server_cert_path "server certificate" "crt pem cer"
    set /p root_cert_path="Path to root certificate file (leave blank if none): "
    if defined root_cert_path call :validate_file_path root_cert_path "root certificate" "crt pem cer"
    set /p intermediate_cert_path="Path to intermediate certificate file (leave blank if none): "
    if defined intermediate_cert_path call :validate_file_path intermediate_cert_path "intermediate certificate" "crt pem cer"
    call :get_input private_key_path "Path to private key file"
    call :validate_file_path private_key_path "private key" "key"
    call :get_input domain_name "Domain name for this certificate"
    call :get_input company_name "Company name"
    set /p organizational_unit="Organizational unit (leave blank if none): "
    call :get_input city "City/locality"
    call :get_input state "State/province"
    call :get_input country_code "Two-letter country code (e.g., US)"
    call :get_input alias "Alias for the certificate in the keystore" "dvsum-gateway-cert"
    call :get_input keystore_pass "Keystore password"
) else if "%ssl_option%"=="4" (
    call :get_input keystore_path "Path to the existing keystore"
    call :validate_file_path keystore_path "keystore" "jks keystore ks"
    call :get_input keystore_pass "Keystore password"
)

endlocal & (
    set "domain_name=%domain_name%"
    set "server_public_ip=%server_public_ip%"
    set "company_name=%company_name%"
    set "organizational_unit=%organizational_unit%"
    set "city=%city%"
    set "state=%state%"
    set "country_code=%country_code%"
    set "alias=%alias%"
    set "keystore_pass=%keystore_pass%"
    set "keystore_path=%keystore_path%"
    set "server_cert_path=%server_cert_path%"
    set "root_cert_path=%root_cert_path%"
    set "intermediate_cert_path=%intermediate_cert_path%"
    set "private_key_path=%private_key_path%"
)
exit /b 0

:get_input
set "input_var=%~1"
set "prompt_text=%~2"
set "default_value=%~3"
:input_loop
if defined default_value (
    set /p "%input_var%=%prompt_text% (%default_value%): "
) else (
    set /p "%input_var%=%prompt_text%: "
)
if "!%input_var%!"=="" (
    if defined default_value (
        set "%input_var%=%default_value%"
    ) else (
        echo The value cannot be empty. Please enter a valid value.
        goto input_loop
    )
)
exit /b 0


:validate_file_path
setlocal EnableDelayedExpansion
set "path_var=%~1"
set "path=!%path_var%!"
set "type=%~2"
set "extensions=%~3"

:validate_file_path_loop
if not exist "!path!" (
    echo Error: The specified %type% file does not exist.
    set /p "path=Please enter a valid path for the %type% file: "
    goto :validate_file_path_loop
)

if defined extensions (
    set "valid_extension="
    for %%e in (%extensions%) do (
        if "!path:~-4!" == ".%%e" set "valid_extension=1"
    )
    if not defined valid_extension (
        echo Error: The %type% file must have one of these extensions: %extensions%
        set /p "path=Please enter a valid path for the %type% file (with correct extension): "
        goto :validate_file_path_loop
    )
)

endlocal & set "%path_var%=%path%"
exit /b 0

:create_keystore_and_csr
set "keystore_path=%domain_name%.p12"
if exist "%keystore_path%" del "%keystore_path%"
echo Existing %domain_name%.p12 deleted.

call :get_ip_address public_ip

"%KEYTOOL%" -genkeypair -alias "%alias%" -keyalg "%encryption_algorithm%" -keysize %key_size% -keystore "%keystore_path%" -storepass "%keystore_pass%" ^
    -dname "CN=%domain_name%, OU=%organizational_unit%, O=%company_name%, L=%city%, ST=%state%, C=%country_code%" ^
    -ext "SAN=dns:localhost,ip:127.0.0.1,ip:%public_ip%" -validity 3650

if !errorlevel! neq 0 (
    echo Failed to create a new key pair. Exiting.
    set "process_successful=false"
    exit /b 1
)

"%KEYTOOL%" -certreq -alias "%alias%" -keystore "%keystore_path%" -storepass "%keystore_pass%" -file "%domain_name%.csr"
if !errorlevel! neq 0 (
    echo Failed to create CSR. Exiting.
    set "process_successful=false"
    exit /b 1
)
exit /b 0

:get_ip_address
set ip_type=%1
set ip_address=

if defined server_public_ip if not "%server_public_ip%"=="" (
    set ip_address=%server_public_ip%
) else (
    for /f "tokens=* delims=" %%a in ('powershell -Command "(Invoke-WebRequest -Uri https://api.ipify.org -UseBasicParsing).Content"') do set ip_address=%%a
)

if "%ip_address%"=="" (
    for /f "tokens=2 delims=:" %%a in ('ipconfig ^| findstr /c:"IPv4 Address"') do (
        set ip_address=%%a
        set ip_address=!ip_address:~1!
    )
)

if "%ip_address%"=="" set ip_address=127.0.0.1

set "%ip_type%=%ip_address%"
exit /b 0

:convert_p12_to_jks
set "jks_path=%domain_name%.jks"
if exist "%jks_path%" del "%jks_path%"
echo Existing %domain_name%.jks deleted.

"%KEYTOOL%" -importkeystore -srckeystore "%keystore_path%" -srcstoretype PKCS12 -destkeystore "%jks_path%" ^
    -deststoretype JKS -srcstorepass "%keystore_pass%" -deststorepass "%keystore_pass%"
if !errorlevel! neq 0 (
    echo Failed to convert keystore. Exiting.
    set "process_successful=false"
    exit /b 1
)
set "keystore_path=%domain_name%.jks"
exit /b 0

:import_certificates
if defined root_cert_path (
    call :delete_and_import "root" "%root_cert_path%"
)
if defined intermediate_cert_path (
    call :delete_and_import "intermediate" "%intermediate_cert_path%"
)
"%KEYTOOL%" -import -alias "%alias%" -keystore "%keystore_path%" -trustcacerts -file "%server_cert_path%" -storepass "%keystore_pass%" -noprompt
if !errorlevel! neq 0 (
    echo Failed to import server certificate.
    set "process_successful=false"
    exit /b 1
)
if "%process_successful%"=="true" (
    echo Certificates imported successfully.
) else (
    echo Certificate import process completed with errors.
)
exit /b 0

:delete_and_import
set "cert_alias=%~1"
set "cert_path=%~2"
"%KEYTOOL%" -list -alias "%cert_alias%" -keystore "%keystore_path%" -storepass "%keystore_pass%" >nul 2>&1
if !errorlevel! equ 0 (
    set /p choice="Alias '%cert_alias%' exists. Do you want to delete it and import a new certificate? (y/n): "
    if /i "!choice!"=="y" (
        "%KEYTOOL%" -delete -alias "%cert_alias%" -keystore "%keystore_path%" -storepass "%keystore_pass%"
        if !errorlevel! neq 0 (
            echo Failed to delete existing certificate with alias '%cert_alias%'.
            set "process_successful=false"
            exit /b 1
        )
    ) else (
        echo Skipping import for alias '%cert_alias%'.
        exit /b 0
    )
)
"%KEYTOOL%" -importcert -alias "%cert_alias%" -file "%cert_path%" -keystore "%keystore_path%" -storepass "%keystore_pass%" -trustcacerts -noprompt
if !errorlevel! neq 0 (
    echo Failed to import certificate with alias '%cert_alias%'.
    set "process_successful=false"
)
exit /b 0

:create_keystore_from_separate_files
set "new_keystore_path=%domain_name%.jks"
"%OPENSSL%" pkcs12 -export -in "%server_cert_path%" -inkey "%private_key_path%" -out temp.p12 -name "%alias%" -passout pass:"%keystore_pass%"
if !errorlevel! neq 0 (
    echo Failed to create temporary PKCS12 file. Exiting.
    set "process_successful=false"
    exit /b 1
)

call :get_ip_address public_ip

"%KEYTOOL%" -importkeystore -srckeystore temp.p12 -srcstoretype PKCS12 -srcstorepass "%keystore_pass%" ^
    -destkeystore "%new_keystore_path%" -deststoretype JKS -deststorepass "%keystore_pass%" ^
    -alias "%alias%" -ext "SAN=dns:localhost,ip:127.0.0.1,ip:%public_ip%" -validity 3650

if !errorlevel! neq 0 (
    echo Failed to create JKS keystore. Exiting.
    del temp.p12
    set "process_successful=false"
    exit /b 1
)

del temp.p12

call :import_additional_certificates

if "%process_successful%"=="true" (
    echo New keystore created successfully with imported certificate and private key for domain: %domain_name%
    set "keystore_path=%new_keystore_path%"
) else (
    echo Keystore creation failed. View error logs
)
exit /b 0

:import_additional_certificates
if defined root_cert_path (
    echo Adding root certificate .......
    "%KEYTOOL%" -importcert -alias root -file "%root_cert_path%" -keystore "%new_keystore_path%" ^
        -storepass "%keystore_pass%" -noprompt
    if !errorlevel! neq 0 (
        echo Failed to import root certificate.
    )
)

if defined intermediate_cert_path (
    echo Adding intermediate certificate .............
    "%KEYTOOL%" -importcert -alias chain -file "%intermediate_cert_path%" -keystore "%new_keystore_path%" ^
        -storepass "%keystore_pass%" -noprompt
    if !errorlevel! neq 0 (
        echo Failed to import intermediate certificate.
    )
)
exit /b 0

:check_keystore_content
    echo Verifying keystore contents:
    "%KEYTOOL%" -list -v -keystore "%keystore_path%" -storepass "%keystore_pass%" -noprompt
exit /b 0

:verify_keystore
if "%process_successful%"=="true" (
    echo.
    set /p verify_choice="Do you want to verify the keystore contents? (y/n): "
    if /i "%verify_choice%"=="y" (
        echo.
        echo Verifying keystore contents:
        "%KEYTOOL%" -list -v -keystore "%keystore_path%" -storepass "%keystore_pass%"
    ) else (
        echo Skipping keystore verification.
    )

    echo.
    echo SSL Configuration Properties for your Java application:
    echo key-store=%keystore_path%
    echo key-store-password=%keystore_pass%
    echo key-store-type=JKS
    echo key-alias=%alias%

    echo.
    echo SSL Configuration completed successfully. Your keystore is ready for use in your Java application.
    echo Please securely store your keystore file and password.
) else (
    echo.
    echo SSL Configuration process completed with errors. Please review the output above for details.
    echo Press Enter to continue...
    pause >nul
)
exit /b 0

:generate_ssl_config_file
(
echo # SSL Configuration Properties
echo # This file contains the necessary configurations for SSL setup and DvSum Gateway service.
echo.
echo # ----------------------------------------
echo # 1. Domain and Company Information
echo # ----------------------------------------
echo domain_name=%domain_name%
echo company_name=%company_name%
echo organizational_unit=%organizational_unit%
echo city=%city%
echo state=%state%
echo country_code=%country_code%
echo server_public_ip=%server_public_ip%
echo.
echo # ----------------------------------------
echo # 2. Keystore Configuration
echo # ----------------------------------------
echo keystore_path=%keystore_path%
echo keystore_pass=%keystore_pass%
echo key_store_type=JKS
echo alias=%alias%
echo.
echo # ----------------------------------------
echo # 3. Certificate Paths
echo # ----------------------------------------
echo server_cert_path=%server_cert_path%
echo private_key_path=%private_key_path%
echo root_cert_path=%root_cert_path%
echo intermediate_cert_path=%intermediate_cert_path%
) > "%SSL_CONFIG_FILE%"

echo Configuration has been updated with the latest values.
echo The updated configuration file is: %SSL_CONFIG_FILE%
exit /b 0

:setup_services
echo.
echo Setting up DvSum Gateway services...
echo ------------------------------------

call :setup_single_service "%gateway_service_name%" "%CD%\DvSum.jar" "DvSum Gateway Service" "com.dvsum.initializer.EmbeddedJetty" "main" "stopJetty"
call :setup_single_service "%gateway_monitor_name%" "%CD%\DvSum_monitor.jar" "DvSum Gateway Monitoring Service" "com.dvsum.starter.SAWSStarter" "main" "stopStarterService"
exit /b 0

REM Generic function to check if a service exists
:check_service_exists
set "service_name=%~1"
sc query "%service_name%" > nul 2>&1
if %errorlevel% equ 0 (
    echo.
    exit /b 0
) else (
    echo Service %service_name% does not exist.
    exit /b 1
)

:setup_single_service
setlocal EnableDelayedExpansion
set "service_name=%~1"
set "jar_file=%~2"
set "description=%~3"
set "start_class=%~4"
set "start_method=%~5"
set "stop_method=%~6"

sc query "%service_name%" > nul 2>&1
if !errorlevel! equ 0 (
    echo %service_name% service already exists. Restarting...
    call :stop_service %service_name%
	call :start_service %service_name%
) else (
	echo.
    echo Installing %service_name% service...
    "%CD%\prunsrv_AMD64.exe" //IS//%service_name% ^
        --DisplayName="%service_name%" ^
        --Description="%description%" ^
        --Install="%CD%\prunsrv_AMD64.exe" ^
        --Jvm="%CD%\jre1.8.0_202\bin\server\jvm.dll" ^
        --StartMode=jvm --StopMode=jvm ^
        --StartClass=%start_class% ^
        --StartMethod=%start_method% ^
        --StopClass=%start_class% ^
        --StopMethod=%stop_method% ^
        --Classpath="%jar_file%" ^
        --Startup=auto ^
        ++JvmOptions=-Djetty.home="%CD%" ^
        --JavaHome="%CD%\jre1.8.0_202" ^
        --LibraryPath="%CD%\jre1.8.0_202\bin" ^
        --LogPath="%CD%\installer.log" 2>>"%CD%\installer.log\error.log"
    
	if !errorlevel! neq 0 (
		echo Failed to install %service_name% service. Check error.log for details here "%CD%\installer.log\error.log".
	) else (
		echo %service_name% service installed successfully.
		timeout /t 5 /nobreak > nul
		"%CD%\prunsrv_AMD64.exe" //ES//%service_name% 2>>"%CD%\installer.log\error.log"
		if !errorlevel! neq 0 (
			echo Failed to start %service_name% service. Check error.log for details here "%CD%\installer.log\error.log".
		) else (
			echo %service_name% service started successfully.
		)
	)
)

REM Verify if the service is actually running
sc query "%service_name%" | find "RUNNING" > nul
if !errorlevel! equ 0 (
    echo %service_name% service is running.
) else (
    echo %service_name% service is not running. Please check the logs for more details.
)

endlocal
exit /b 0

:setup_python_service
setlocal
set "service_name=%gateway_python_service_name%"

REM Determine the appropriate NSSM executable (64-bit or 32-bit) from the current directory
set "nssm_executable="
if defined ProgramFiles(x86) (
    set "nssm_executable=%~dp0nssm_AMD64.exe"  REM 64-bit system
) else (
    set "nssm_executable=%~dp0nssm_x86.exe"  REM 32-bit system
)

REM Check if Python is installed
set "PYTHON_EXECUTABLE="
FOR /F "delims=" %%i IN ('where python 2^>nul') DO (
    SET "PYTHON_EXECUTABLE=%%i"
    IF NOT "%%i"=="C:\Users\%USERNAME%\AppData\Local\Microsoft\WindowsApps\python.exe" GOTO FoundPython
)

REM If Python is not installed or is only the placeholder
if not defined PYTHON_EXECUTABLE (
    echo Python is not installed on this system.
    echo Please install Python manually to use the DvSum Python Service.
    exit /b 1
)

if "%PYTHON_EXECUTABLE%"=="C:\Users\%USERNAME%\AppData\Local\Microsoft\WindowsApps\python.exe" (
    echo Python placeholder detected. Python is not properly installed.
    echo Please install Python manually to use the DvSum Python Service.
    exit /b 1
)

:FoundPython
echo Python executable found: "%PYTHON_EXECUTABLE%"

REM Check if the service already exists using NSSM
"%nssm_executable%" status "%service_name%" > nul 2>&1
if !errorlevel! equ 0 (
    echo %service_name% service already exists. Stopping service...
    "%nssm_executable%" stop "%service_name%"
    "%nssm_executable%" remove "%service_name%" confirm
)

REM Install Python dependencies
echo Installing Python dependencies from "%REQUIREMENTS_FILE%"...
"%PYTHON_EXECUTABLE%" -m pip install -r "%REQUIREMENTS_FILE%" || (
    echo Failed to install Python dependencies.
    exit /b 1
)

REM Create the Python service using NSSM
echo Creating the %service_name% service using NSSM...
"%nssm_executable%" install "%service_name%" "%PYTHON_EXECUTABLE%" "%FLASK_APP%"
"%nssm_executable%" set "%service_name%" Start SERVICE_AUTO_START

REM Check if the service creation was successful
if !errorlevel! neq 0 (
    echo Failed to create %service_name% service.
    exit /b 1
)

REM Configure the service to use the appropriate Python environment and log file
"%nssm_executable%" set "%service_name%" AppParameters "\"%FLASK_APP%\""
"%nssm_executable%" set "%service_name%" AppDirectory "\"%CD%\python_service\""
"%nssm_executable%" set "%service_name%" AppStdout "\"%CD%\python_service\logs\python_service_stdout.log\""
"%nssm_executable%" set "%service_name%" AppStderr "\"%CD%\python_service\logs\python_service_stderr.log\""
"%nssm_executable%" set "%service_name%" Description "Python Flask service for DvSum"

echo %service_name% service setup completed.
"%nssm_executable%" start "%service_name%"
endlocal
exit /b 0


:start_python_service
setlocal
set "service_name=%~1"

REM Determine the appropriate NSSM executable (64-bit or 32-bit) from the current directory
set "nssm_executable="
if defined ProgramFiles(x86) (
    set "nssm_executable=%CD%\nssm_AMD64.exe"  REM 64-bit system
) else (
    set "nssm_executable=%CD%\nssm_x86.exe"  REM 32-bit system
)

REM Check if the service exists using NSSM
"%nssm_executable%" status "%service_name%" > nul 2>&1
if !errorlevel! neq 0 (
    echo.
    echo Service "%service_name%" does not exist.
    exit /b 1
)

echo.
echo Starting "%service_name%" service...
"%nssm_executable%" start "%service_name%"

REM Check if the service started successfully
if !errorlevel! equ 0 (
    echo "%service_name%" service started successfully.
) else (
    echo Failed to start "%service_name%" service.
    exit /b 1
)

endlocal
exit /b 0



:stop_python_service
setlocal
set "service_name=%~1"

REM Determine the appropriate NSSM executable (64-bit or 32-bit) from the current directory
set "nssm_executable="
if defined ProgramFiles(x86) (
    set "nssm_executable=%~dp0nssm_AMD64.exe"  REM 64-bit system
) else (
    set "nssm_executable=%~dp0nssm_x86.exe"  REM 32-bit system
)

REM Check if the service exists using NSSM
call "%nssm_executable%" status "%service_name%" > nul 2>&1
if errorlevel 1 (
    echo Service "%service_name%" does not exist.
    exit /b 1
)

REM Stop the service using NSSM
echo.
echo Stopping "%service_name%" service...
call "%nssm_executable%" stop "%service_name%" > nul 2>&1

REM Verify if the service stopped successfully
call "%nssm_executable%" status "%service_name%" > "%TEMP%\service_status.txt"
findstr /I "STOPPED" "%TEMP%\service_status.txt" > nul 2>&1

if errorlevel 1 (
    del "%TEMP%\service_status.txt"
    exit /b 1
)

del "%TEMP%\service_status.txt"
echo "%service_name%" service stopped successfully.
endlocal
exit /b 0



:uninstall_python_service
setlocal
set "service_name=%~1"

REM Determine the appropriate NSSM executable (64-bit or 32-bit) from the current directory
set "nssm_executable="
if defined ProgramFiles(x86) (
    set "nssm_executable=%~dp0nssm_AMD64.exe"  REM 64-bit system
) else (
    set "nssm_executable=%~dp0nssm_x86.exe"  REM 32-bit system
)

echo Uninstalling "%service_name%" service...

REM Check if the service exists using NSSM
call "%nssm_executable%" status "%service_name%" > nul 2>&1
if errorlevel 1 (
    echo Service "%service_name%" is already uninstalled or does not exist.
    exit /b 0
)

REM Stop the service first
call :stop_python_service "%service_name%"

REM Remove the service using NSSM
call "%nssm_executable%" remove "%service_name%" confirm > nul 2>&1

REM Verify if the service was successfully removed
call "%nssm_executable%" status "%service_name%" > nul 2>&1
if errorlevel 1 (
    echo Service "%service_name%" uninstalled successfully.
    exit /b 0
) else (
    echo Failed to uninstall "%service_name%". Manual intervention may be required.
    exit /b 1
)

endlocal
exit /b 0



:start_services

call :start_service "%gateway_service_name%"
call :start_service "%gateway_monitor_name%"
call :start_python_service "%gateway_python_service_name%"
exit /b 0

:start_service
setlocal
set "service_name=%~1"
call :check_service_exists "%service_name%"
echo Starting %service_name% service...
timeout /t 5 /nobreak > nul
"%CD%\prunsrv_AMD64.exe" //ES//%service_name%
if !errorlevel! equ 0 (
    echo %service_name% service started successfully.
) else (
    echo Failed to start %service_name% service.
)
endlocal
exit /b 0

:stop_services
call :stop_service "%gateway_monitor_name%"
call :stop_service "%gateway_service_name%"
call :stop_python_service "%gateway_python_service_name%"
exit /b 0

:stop_service
set "service_name=%~1"
call :check_service_exists "%service_name%"
REM If service didn't stop, force kill the process
for /f "tokens=2 delims=:" %%a in ('sc queryex "%service_name%" ^| findstr "PID"') do (
    set "pid=%%a"
)

if not "!pid!"=="" if !pid! neq 0 (
	echo Stopping %service_name% service...
    taskkill /F /PID !pid!
) else (
    echo %service_name% is already stopped.
)

exit /b 0

:uninstall_service
setlocal EnableDelayedExpansion
set "service_name=%~1"
echo Uninstalling %service_name% service...

REM Check if the service still exists
sc query "%service_name%" > nul 2>&1
if !errorlevel! neq 0 (
    echo Service %service_name% is already uninstalled.
    exit /b 0
) else (


	REM Stop the service first
	call :stop_service "%service_name%"



	REM Attempt to uninstall the service
	"%CD%\prunsrv_AMD64.exe" //DS//%service_name%
	if !errorlevel! neq 0 (
		echo Failed to initiate uninstall for service %service_name%.
		exit /b 1
	)

	echo Service %service_name% uninstall initiated.

	REM Wait for the service to be fully removed (up to 60 seconds)
	for /l %%i in (1,1,60) do (
		sc query "%service_name%" > nul 2>&1
		if !errorlevel! neq 0 (
			echo Service %service_name% uninstalled successfully.
			echo.
			timeout /t 5 /nobreak > nul
			exit /b 0
		)
		timeout /t 1 /nobreak > nul
	)

	echo Warning: Service %service_name% may not be fully uninstalled.
	echo Attempting forceful removal...

	REM Attempt forceful removal using sc delete
	sc delete "%service_name%" > nul 2>&1
	if !errorlevel! equ 0 (
		echo Service %service_name% forcefully removed.
		timeout /t 5 /nobreak > nul
		exit /b 0
	)

	echo Failed to forcefully remove %service_name%. Manual intervention may be required.
	echo Please check the following:
	echo 1. Open Services.msc and verify if the service still exists.
	echo 2. If it exists, try to stop and delete it manually.
	echo 3. Check for any running processes related to the service and end them.
	echo 4. Reboot the system if the service persists.
)
endlocal
exit /b 1

:uninstall_services
echo.
echo Uninstalling DvSum Gateway services...
echo --------------------------------------
call :uninstall_service "%gateway_monitor_name%"
call :uninstall_service "%gateway_service_name%"
call :uninstall_python_service "%gateway_python_service_name%"
exit /b 0

:restart_services
call :stop_services
call :start_services
exit /b 0

:check_service_status
echo.
echo Checking DvSum Gateway services status...
call :check_single_service_status "%gateway_service_name%"
call :check_single_service_status "%gateway_monitor_name%"
call :check_single_service_status "%gateway_python_service_name%"
exit /b 0

:check_single_service_status
setlocal
set "service_name=%~1"
echo.
echo Status for %service_name%:
echo -------------------------------------
call :check_service_exists "%service_name%"
sc queryex "%service_name%" | findstr /C:"SERVICE_NAME" /C:"STATE" /C:"PID"
echo.

endlocal
exit /b 0

:update_configurations
setlocal EnableDelayedExpansion
set "config_file=configuration.properties"
if not exist "%config_file%" (
    echo Error: %config_file% not found.
    exit /b 1
)
REM Read existing values
set "api.key="
set "port="
set "max.database.connection.allowed="
set "python.service.port="
for /f "tokens=1,* delims==" %%a in ('type "%config_file%"') do (
    if "%%a"=="api.key" set "api.key=%%b"
    if "%%a"=="port" set "port=%%b"
    if "%%a"=="max.database.connection.allowed" set "max.database.connection.allowed=%%b"
    if "%%a"=="python.service.port" set "python.service.port=%%b"
)
REM Remove leading/trailing spaces
set "api.key=%api.key: =%"
set "port=%port: =%"
set "max.database.connection.allowed=%max.database.connection.allowed: =%"
set "python.service.port=%python.service.port: =%"

REM Update API key
if defined cmd_api_key (
    set "api.key=!cmd_api_key!"
) else if "%interactive%"=="true" (
    if not "!api.key!"=="" (
        echo Current API Key: !api.key!
        set /p "update_api_key=Do you want to update the API Key? (Y/N): "
        if /i "!update_api_key!"=="Y" (
            set /p "new_api_key=Enter new DvSum API Key: "
            set "api.key=!new_api_key: =!"
        )
    ) else (
        set /p "api.key=Enter DvSum API Key: "
        set "api.key=!api.key: =!"
    )
)
if "!api.key!"=="" (
    echo Error: API Key cannot be empty.
    exit /b 1
)
REM Update server port
if defined cmd_port (
    set "port=!cmd_port!"
) else if "%interactive%"=="true" (
    if not "!port!"=="" (
        echo Current Server Port: !port!
        set /p "update_port=Do you want to update the Server Port? (Y/N): "
        if /i "!update_port!"=="Y" (
            call :get_available_port
        )
    ) else (
        call :get_available_port
    )
)
if "!port!"=="" (
    echo Error: Server Port cannot be empty.
    exit /b 1
)
REM Update max database connections
if defined cmd_max_db_conn (
    set "max.database.connection.allowed=!cmd_max_db_conn!"
) else if "%interactive%"=="true" (
    if not "!max.database.connection.allowed!"=="" (
        echo Current Max Database Connections: !max.database.connection.allowed!
        set /p "update_max_db_conn=Do you want to update the Max Database Connections? (Y/N): "
        if /i "!update_max_db_conn!"=="Y" (
            call :get_max_db_connections
        )
    ) else (
        call :get_max_db_connections
    )
)
if "!max.database.connection.allowed!"=="" (
    echo Error: Max Database Connections cannot be empty.
    exit /b 1
)

REM Update python service port
if defined cmd_python_service_port (
    set "python.service.port=!cmd_python_service_port!"
) else if "%interactive%"=="true" (
    if not "!python.service.port!"=="" (
        echo Current Python Service Port: !python.service.port!
        set /p "update_python_service_port=Do you want to update the Python Service Port? (Y/N): "
        if /i "!update_python_service_port!"=="Y" (
            call :get_available_python_service_port
        )
    ) else (
        call :get_available_python_service_port
    )
)
if "!python.service.port!"=="" (
    echo Error: Python Service Port cannot be empty.
    exit /b 1
)

REM Update the configuration file
(
    for /f "tokens=1,* delims==" %%a in ('type "%config_file%"') do (
        if "%%a"=="api.key" (
            echo api.key=!api.key!
        ) else if "%%a"=="port" (
            echo port=!port!
        ) else if "%%a"=="max.database.connection.allowed" (
            echo max.database.connection.allowed=!max.database.connection.allowed!
        ) else if "%%a"=="python.service.port" (
            echo python.service.port=!python.service.port!
        )  else (
            echo %%a=%%b
        )
    )
) > "%config_file%.tmp"
REM Add properties if they didn't exist before
findstr /m "^api.key=" "%config_file%.tmp" >nul
if %errorlevel% neq 0 echo api.key=!api.key!>> "%config_file%.tmp"
findstr /m "^port=" "%config_file%.tmp" >nul
if %errorlevel% neq 0 echo port=!port!>> "%config_file%.tmp"
findstr /m "^max.database.connection.allowed=" "%config_file%.tmp" >nul
if %errorlevel% neq 0 echo max.database.connection.allowed=!max.database.connection.allowed!>> "%config_file%.tmp"
findstr /m "^python.service.port=" "%config_file%.tmp" >nul
if %errorlevel% neq 0 echo python.service.port=!python.service.port!>> "%config_file%.tmp"
move /y "%config_file%.tmp" "%config_file%" > nul
endlocal
exit /b 0

:get_available_port
set /p "port=Enter Server Port: "
call :check_port_availability "!port!"
if !errorlevel! neq 0 (
    echo Port !port! is not available. Please choose a different port.
    goto :get_available_port
)
exit /b 0

:check_port_availability
netstat -an | find ":%~1 " > nul
if %errorlevel% equ 0 (
    exit /b 1
) else (
    exit /b 0
)

:get_available_python_service_port
set /p "python.service.port=Enter Python Service Port: "
REM Check if the selected port matches the restricted port variable
if "!python.service.port!"=="!port!" (
    echo Port !python.service.port! cannot be used. It matches the restricted port !port!.
    goto :get_available_python_service_port
)
call :check_port_availability "!python.service.port!"
if !errorlevel! neq 0 (
    echo Port !python.service.port! is not available. Please choose a different port.
    goto :get_available_python_service_port
)
exit /b 0


:get_max_db_connections
:retry_max_db_conn
set /p "max_db_conn=Enter Max Database Connections (must be a positive integer): "
echo %max_db_conn%| findstr /r "^[1-9][0-9]*$" >nul
if %errorlevel% neq 0 (
    echo Invalid input. Please enter a positive integer.
    goto :retry_max_db_conn
)
set "max.database.connection.allowed=%max_db_conn%"
exit /b 0



:validate_ssl_option
if "%ssl_option%" neq "1" if "%ssl_option%" neq "2" if "%ssl_option%" neq "3" if "%ssl_option%" neq "4" (
    echo Error: Invalid SSL option. Must be 1, 2, 3, or 4.
    exit /b 1
)
exit /b 0

:verify_existing_keystore
echo Verifying existing keystore...
echo Keystore path: %keystore_path%
"%KEYTOOL%" -list -v -keystore "%keystore_path%" -storepass "%keystore_pass%"
if !errorlevel! neq 0 (
    echo Error: Failed to verify the existing keystore. Please check the path and password.
    exit /b 1
)
echo Existing keystore verified successfully.
echo.
echo Please review the keystore information above to ensure it contains the correct certificate and private key.
echo.
set /p "confirm=Is this the correct keystore? (Y/N): "
if /i "%confirm%" neq "Y" (
    echo Keystore verification cancelled by user.
    exit /b 1
)
exit /b 0

:backup_keystore
set "backup_dir=ssl_tools\keystore_backups\%date:~-4,4%%date:~-10,2%%date:~-7,2%_%time:~0,2%%time:~3,2%%time:~6,2%"
mkdir "%backup_dir%" 2>nul
for %%f in ("%domain_name%.jks" "%domain_name%.p12" "%domain_name%.csr") do (
    if exist "%%f" (
        echo Backing up existing %%~nxf
        copy "%%f" "%backup_dir%\"
    )
)
echo Keystore files backed up to: %backup_dir%
exit /b 0

:display_help
echo DvSum Gateway Manager
echo ---------------------
echo This script manages the DvSum Gateway, including installation, SSL configuration,
echo and service control.
echo.
call :display_usage
exit /b 0


:display_usage
echo.
echo Usage: dvsum_gateway_setup.bat [OPTIONS] [ACTION]
echo.
echo OPTIONS:
echo   "--gateway-service-name=<name>"         Specify custom name for the gateway service
echo   "--gateway-monitor-name=<name>"         Specify custom name for the monitor service
echo   "--gateway-python-service-name=<name>"  Specify custom name for the python service
echo   "--non-interactive"                     Run in non-interactive mode
echo   "--ssl-option=<1|2|3|4>"                Specify SSL configuration option
echo   "--encryption-algorithm=<algo>"         Specify encryption algorithm (RSA, DSA, EC)
echo   "--key-size=<size>"                     Specify key size (depends on algorithm)
echo   "--verify-keystore"                     Verify the contents of the keystore
echo   "--check-network"                       Verify Outbound Access to DvSum Web Application
echo   "--help"                                Display the help for using this script
echo.
echo ACTIONS:
echo   start                   Start the DvSum Gateway services
echo   stop                    Stop the DvSum Gateway services
echo   restart                 Restart the DvSum Gateway services
echo   status                  Check the status of the DvSum Gateway services
echo   uninstall               Uninstall the DvSum Gateway services
echo   install                 Perform full installation and configuration (default)
echo.
echo Examples:
echo   dvsum_gateway_setup.bat --gateway-service-name=CustomGateway --gateway-monitor-name=CustomMonitor --gateway-python-service-name=CustomPythonService install
echo   dvsum_gateway_setup.bat --ssl-option=1 --non-interactive
echo   dvsum_gateway_setup.bat --ssl-option=2 --encryption-algorithm=RSA --key-size=2048 install
echo   dvsum_gateway_setup.bat start
echo   dvsum_gateway_setup.bat stop
echo   dvsum_gateway_setup.bat status
exit /b 0

:display_completion_message
echo.
echo Debug: process_successful value is !process_successful!
if "!process_successful!"=="true" (
    call :display_success_message
) else (
    call :display_error_message
)
echo.
if "!interactive!"=="true" (
    pause
)
exit /b 0

:display_success_message
echo Debug: Displaying success message
echo ------------------------------------------------------------
echo [OK] DvSum Gateway Installation and Configuration Complete!
echo ------------------------------------------------------------
echo Great news! The setup process has finished successfully.
echo.
echo Here's what we've accomplished:
echo 1. SSL has been configured for secure communication
echo 2. DvSum Gateway services have been installed
echo 3. All necessary services have been started
echo.
echo Next steps:
echo . Verify that the Gateway is accessible at: https://^<your-domain^>:^<port^>
echo . Check the logs at ./logs/SAWS.log for any important messages
echo . Refer to the DvSum documentation for post-installation tasks
echo.
echo If you encounter any issues or have questions, please contact
echo DvSum support at support@dvsum.com
echo.
echo Thank you for choosing DvSum Gateway!
exit /b 0

:display_error_message
echo Debug: Displaying error message
echo ------------------------------------------------------------
echo [ERROR] DvSum Gateway Installation and Configuration Incomplete
echo ------------------------------------------------------------
echo We encountered some issues during the setup process.
echo.
echo Here's what you should do:
echo 1. Review the error messages above for specific details
echo 2. Check the logs at ./logs/SAWS.log for additional information
echo 3. Ensure all prerequisites are met (refer to documentation)
echo 4. Try running the installation script again
echo.
echo If the problem persists, please contact DvSum support:
echo . Email: support@dvsum.com
echo . Include the log file and any error messages you've seen
echo.
echo We apologize for the inconvenience and are here to help!
exit /b 0


:end
endlocal
exit /b 0